#####################################   UFPROC.PY   ##################################################
#                                                                                                       
# ufproc.py is a python program to process any ultrafast nmr acquisition.(single shot, interleaved, J-Resolved...)                          
#                                                                                                       
# The script needs:                                                                                     
#        * copy ufproc.py to TOPSPIN/exp/stan/nmr/py/user/                                              
#
# execution by topspin command:  ufproc
#
# Arguments (can be combined):
#
#       help: 
#       ufproc help: List of valid options (arguments) for ufproc
#
#       type of acquisition:
#       default: One shot acquisition (TD 1 = 1)
#       ufproc inter : Interleaved acquisition
#       ufproc jres : J-resolved acquisition
#
#	time offset:
#       default: program will calculate and apply an offset
#       ufproc x :will apply the user defined time offset (x)
#       ufproc 0 :will not apply any offset (x=0)#
#       ufproc dz :To use in case of high amplitude UF board effect 
#                  will remove both 1/8th extremities of the spectra for the time offset calculation
#
#       Shearing correction:
#       default: program will not apply any shearing correction
#       ufproc shon: will force program to apply shearing correction
#       
#       BaseLine correction:
#       default: program do not apply the baseline correction
#       ufproc abson: will apply the baseline correction
#
#       Miscellaneous:
#       ufproc save: will save the 2D map of both positive and negative gradients (grad+ and grad-)
#       ufproc + :will process only acquisition during positive gradients
#		rot: apply a rotation F1/F2 of the 2D map 
#
# Compatible with Bruker Topspin 2.x; Topspin 3.x; Topspin 4.x
# Not compatible with Topspin 1.x (a special version exists for 1.x)

#
# Version: 2019-01-09
#
# Author: Benoit CHARRIER, Laboratoire CEISAM, Nantes
# Mail: Benoit.Charrier@univ-nantes.fr
# Modified by: Laetitia ROUGER, Laboratoire CEISAM, Nantes
# Mail: Laetitia.Rouger@univ-nantes.fr
# Modified by: Bertrand PLAINCHONT, Laboratoire CEISAM, Nantes
# Mail: Bertrand.Plainchont@univ-nantes.fr
# Supervisor: Patrick GIRAUDEAU,  Laboratoire CEISAM, Nantes
# Mail: Patrick.Giraudeau@univ-nantes.fr
########################################################################################################

from de.bruker.nmr.prsc.util import DataChecks
from de.bruker.nmr.jutil.xwin import UtilXwin
import os.path
import time
import shutil
import struct

t1 = time.clock()

#___________________________________________________FUNCTIONS__________________________________________________


def deltree(rep):
        try:    shutil.rmtree(rep)
        except:
                msg_deltree(rep)
                EXIT()
def copyfile(path,source,dest):
        try:   shutil.copyfile(path+'/'+source,path+'/'+dest)
        except:
                msg_copyfile(path,source)
                EXIT()

def is_int(s):
        try: 
                int(s)
                return 1
        except ValueError: return 0

def disp(n,e,p,d,u):
        if u=='nouser':RE([n,e,p,d],'y') #topspin 3
        else:   RE([n,e,p,d,u],'y')      #topspin 2        

def fmt(dtyp,bytord): #define size and formar to read/write binaries with struct class.
        if dtyp==0 or dtyp=='proc':
                size=4
                typ='i'
        if dtyp==2:
                size=8
                typ='d'
        if bytord==1:  order='>'
        if bytord==0:  order='<'
        fmt_r=order+typ
        fmt_w=order+'%d'+typ
        return (size, fmt_r, fmt_w)

def write_bin(dst,fmt_w,li): dst.write( struct.pack(fmt_w % len(li), *li))

def read_bin(fmt_r,li):return  struct.unpack(fmt_r, li)

def linreg(X, Y): #linear regression for shearing correction
        if len(X) != len(Y):  raise ValueError, 'unequal length'
        N = len(X)
        Sx = Sy = Sxx = Syy = Sxy = 0.0
        for x, y in map(None, X, Y):
                Sx = Sx + x
                Sy = Sy + y
                Sxx = Sxx + x*x
                Syy = Syy + y*y
                Sxy = Sxy + x*y
        det = Sxx * N - Sx * Sx
        a, b = (Sxy * N - Sy * Sx)/det, (Sxx * Sy - Sx * Sxy)/det
        meanerror = residual = 0.0
        for x, y in map(None, X, Y):
                meanerror = meanerror + (y - Sy/N)**2
                residual = residual + (y - a * x - b)**2
        RR = 1 - residual/meanerror
        ss = residual / (N-2)
        Var_a, Var_b = ss * N / det, ss * Sxx / det
        return a, b, RR


def edc2(dir2, user2, name2, expno2, procno2):
	nmrdatapars = DataChecks.getNMRDataOfSelectedFramePrintMsg().getProperties()
	nmrdatapars.setProperty("CURDAT2", UtilXwin.mkXwinDataPath(dir2, user2, name2, expno2, procno2))

def shearing(data_rr,data_ii,SIx,SIy):

        MAX=[0,0,0]
        mod=[]
        for y in xrange(SIy):
                mod.append([])
                mod[y].extend([int(abs(complex(r,i))) for r,i in zip( data_rr[y*SIx:(y+1)*SIx], data_ii[y*SIx:(y+1)*SIx]  )])

        ZONE=range(SIx/DZ,SIx*(DZ-1)/DZ)
        for y in xrange(SIy):
                for x in ZONE:
                        if mod[y][x]>MAX[2]:  MAX = x, y, mod[y][x] #max of all 2D map not in DEAD_ZONE
	
        
        Mx=MAX[0]
        My=MAX[1]
        Mz=MAX[2]
        
        #caculate the ratio max/average in order to decide whether the shearing correction will be reliable.
        s=0
        for val in mod[My]:
                s+=val
        average=s/SIx
        ratio=average*100.0/Mz
        
        if (ratio>15 and arg['shear']!='shon'):
                sys.stdout.write('program has disabled the shearing correction\r\n')
                return 0,'nolist'
        else:
                sys.stdout.write('ratio='+str(ratio)+'<15\r\n')
                # follow maxima along F1 inside a moving range initially centered at Mx.
                # provide a list of index of maxima above the threshold and for each row.
                threshold=int(Mz/4)
                range_x=int(SIx/128)
                
                index=cond=s=0
                start=Mx-range_x
                if start<0:start=1
                stop=Mx+range_x
                if stop>SIx:stop=SIx-2
                list_shift_top=[]
                list_y_top=[]
                list_shift_bot=[]
                list_y_bot=[]
                list_shift=[]
                list_y=[]
                for y in xrange(My,SIy,1):
                        maxi=threshold
                        cond=0
                        for x in xrange(start,stop):
                                if mod[y][x]>maxi:
                                        if mod[y][x]>mod[y][x-1] and mod[y][x]>=mod[y][x+1]:
                                                maxi=mod[y][x]
                                                index=x
                                                cond=1
                        if cond==1:
                                list_shift_top.append(Mx-index)
                                list_y_top.append(y)
                                start=index-range_x
                                if start<0:start=1
                                stop=index+range_x
                                if stop>SIx:stop=SIx-2

                index=cond=s=0
                start=Mx-range_x
                if start<0:start=1
                stop=Mx+range_x
                if stop>SIx:stop=SIx-2
                for y in xrange(My-1,-1,-1):
                        maxi=threshold
                        cond=0
                        for x in xrange(start,stop):
                                if mod[y][x]>maxi:
                                        if mod[y][x]>mod[y][x-1] and mod[y][x]>=mod[y][x+1]:
                                                maxi=mod[y][x]
                                                index=x
                                                cond=1
                        if cond==1:
                                list_shift_bot.append(Mx-index)
                                list_y_bot.append(y)
                                start=index-range_x
                                if start<0:start=1
                                stop=index+range_x
                                if stop>SIx:stop=SIx-2
                                
                list_shift_bot.reverse()
                list_y_bot.reverse()
                list_shift=list_shift_bot+list_shift_top
                list_y=list_y_bot+list_y_top
                
                n=len(list_shift)
                delta=max(list_shift)-min(list_shift)
                delta100=100.0*delta/SIx
                sys.stdout.write('shearing = '+str(delta100)+'%\r\n')

                #decision of applying shearing or not
                if delta100<=0.1:
                        sys.stdout.write('no shearing\r\n')
                        return 0,[]
                        
                elif (n==SIy and arg['shear']!='shon'):
                        sys.stdout.write('shearing correction will be applied\r\n')
                        return 1,list_shift[:]
                
                elif (n==SIy and arg['shear']=='shon'):
                        sys.stdout.write('program force to the shearing correction\r\n')
                        return 1,list_shift[:]
                        
                elif (n<=SIy/4 and arg['shear']!='shon'):
                        sys.stdout.write('program has disabled the shearing correction\r\n')
                        return 0,[]
                        
                elif (n!=SIy and n>SIy/4) or (n<=SIy/4 and arg['shear']=='shon'): #linear regression to complete the list of shifts.
                        if n<=SIy/4:sys.stdout.write('program force to the shearing correction\r\n')
                        a,b,RR=linreg(list_y,list_shift)
                        for y in xrange(SIy):
                                if y not in list_y:
                                        shift=int(round(a*y+b))
                                        list_shift.insert(y,shift)

                        sys.stdout.write('linear reg with '+str(n)+'/'+str(SIy)+' points\r\n')
                        return 1,list_shift[:]

def peak_picking(li,RANGE,threshold):
	PEAK=[ x for x in RANGE if li[x]>threshold and li[x]>li[x-1] and li[x]>=li[x+1] ]
        return PEAK[:]

def offset(projp_bin,projn_bin,MAXp,MAXn,SIx,fmt_r_jp):
        projp=[]
        projn=[]

        fmt=fmt_r_jp[:1]+str(SIx)+fmt_r_jp[1:]

        projp= list(read_bin(fmt,projp_bin))
        projn= list(read_bin(fmt,projn_bin))

        Ip=projp.index(MAXp)

        ZONE=range(SIx/DZ,SIx*(DZ-1)/DZ) 
        MOV_ZONE=range(Ip-SIx/8,Ip+SIx/8)
       
        if Ip not in ZONE: #find another peak and change mov_zone
                while 1:
                        MAXp,Ip = max([(projp[i],i) for i in ZONE])
                        if Ip!=ZONE[0] and Ip!=ZONE[-1]:break
                        if Ip==ZONE[0]:ZONE.pop(0)
                        elif Ip==ZONE[-1]:ZONE.pop()
                MOV_ZONE=range(Ip-SIx/8,Ip+SIx/8)

        RANGE = [x for x in MOV_ZONE if x in ZONE]
        sys.stdout.write('RANGE = ['+str(RANGE[0])+'-'+str(RANGE[-1])+']'+chr(10))
        sys.stdout.write('grad+ refpeak, Ip ='+str(Ip)+chr(10))
        
        threshold=int(MAXp*0.75)            
        
        PEAK=peak_picking(projn,RANGE,threshold)
        n=len(PEAK)

        sys.stdout.write('try1: n peaks = '+str(n)+chr(10))

        if n==0:
                MOV_ZONE=range(Ip-SIx/4,Ip+SIx/4)
                RANGE = [x for x in MOV_ZONE if x in ZONE]
                i=2
                while len(PEAK)==0:
                        threshold=int(MAXp/i)
                        PEAK=peak_picking(projn,RANGE,threshold)
                        i+=1
                n=len(PEAK)
                sys.stdout.write('try'+str(i-1)+': n peaks = '+str(n)+'\r\n')
        if n==1:
                shift=Ip-PEAK[0]
        if n>1:
                list_shift=[]
                for x in PEAK:
                        list_shift.append(Ip-x)
                        
                int_diff=[]
                for shift in list_shift:
                        projn2=[]
                        s=0
                        if shift>=0:
                                projn2=[0]*shift+projn[0:SIx-shift]
                        if shift<0:
                                projn2=projn[abs(shift):]+[0]*abs(shift)
                        for valp,valn in map(None,projp,projn2):
                                s+=abs(valp-valn)/10000
                        int_diff.append(s)
                shift=list_shift[int_diff.index(min(int_diff))]
        sys.stdout.write('Time offset: '+str(shift)+'/'+str(SIx)+' points\r\n')
        return shift


def shift_corrections(src,X,Y,SIx,SIy,fmt_w_gp,shift,list_shift):
        # reading 2rr (blocks)
        _2rr = open(src+'/2rr','rb+')
        _2rr_lin = open(src+'/2rrlin','wb+')
        
        # linearisation of 2rr
        _2rr.seek(0)
        Bx=SIx/X    
        By=SIy/Y
        block=X*size
	XY=X*Y
        for by in xrange(By):
                for y in xrange(Y):
                        for bx in xrange(Bx):
                                _2rr.seek((XY*(bx+by*Bx)+X*y)*size)
                                _2rr_lin.write(_2rr.read(block))

        
        # writing 2rr linearized + offset and shearing correction (need to modify status xdim )
        _2rr_lin.seek(0)
        _2rr.close()
        os.remove(src+'/2rr')
        _2rr = open(src+'/2rr','wb')

        # *********************case time offset only********************************
	if list_shift==[]: 
                #list_shift=[0]*SIy
		if shift>0:
			zeros=shift*[0]
			a=SIx*size
			c=(SIx-shift)*size
			for y in xrange(SIy):				
				write_bin(_2rr,fmt_w_gp,zeros)
				_2rr_lin.seek(a*y)
				_2rr.write( _2rr_lin.read(c) )
		if shift<0:
			toshift=abs(shift)
			zeros=toshift*[0]
			a=SIx*size
			b=toshift*size
			c=(SIx-toshift)*size
			for y in xrange(SIy):
				_2rr_lin.seek(a*y+b )
				_2rr.write( _2rr_lin.read(c) )
				write_bin(_2rr,fmt_w_gp,zeros)	
		
	# **************case time offset + shearing correction**********************
	else:	
		mult=int(SIy/len(list_shift))

		for y in xrange(SIy):
			toshift=shift+list_shift[int(y/mult)]
			if toshift==0:
			       _2rr_lin.seek(y*SIx*size)
			       _2rr.write(_2rr_lin.read(SIx*size)) 
			elif toshift>0:
				write_bin(_2rr,fmt_w_gp,toshift*[0])
				_2rr_lin.seek(y*SIx*size)
				_2rr.write( _2rr_lin.read((SIx-toshift)*size) )
			elif toshift<0:
				_2rr_lin.seek( (y*SIx+abs(toshift))*size )
				_2rr.write( _2rr_lin.read((SIx-abs(toshift))*size) )
				write_bin(_2rr,fmt_w_gp,abs(toshift)*[0])
        
	_2rr.close()
        _2rr_lin.close()




#_____________________________________________________MESSAGES______________________________________________________
        
def msg_badarg(badarg):
	html='<html>"<b>'+badarg+'</b>" is not a valid argument for "ufproc" <br>'
	html+='Valid arguments are: <br><br>'
	html+='<table><tr><td><font color="red">ufproc help</font> :</td><td>List of valid options (arguments) for ufproc</td></tr>'
	html+='<tr><td><font color="red">ufproc inter</font> :</td><td> Interleaved acquisition</td></tr>'
	html+='<tr><td><font color="red">ufproc jres</font> :</td><td> J-resolved acquisition</td></tr>'
	html+='<tr><td><font color="red">ufproc x</font> :</td><td>will apply the user defined time offset (x)</td></tr>'
	html+='<tr><td><font color="red">ufproc 0</font> :</td><td>will not apply any offset (x=0)</td></tr>'
	html+='<tr><td><font color="red">ufproc shon</font> :</td><td> will force program to apply shearing correction</td></tr>'
	html+='<tr><td><font color="red">ufproc abson</font> :</td><td>will apply the user defined baseline correction</td></tr>'
	html+='<tr><td><font color="red">ufproc dz</font> :</td><td>To use in case of high amplitude UF board effect<br>will remove both 1/8th extremities of the spectra for the time offset calculation</td></tr>'
	html+='<tr><td><font color="red">ufproc save</font> :</td><td> will save the 2D map of both positive and negative gradients (grad+ and grad-)</td></tr>'
	html+='<tr><td><font color="red">ufproc rot</font> :</td><td> will apply a rotation F1/F2 of the 2D map</td></tr>'
	html+='<tr><td><font color="red">ufproc +</font> :</td><td>will process only acquisition during positive gradients</td></tr></table><br>'
	html+='Program exiting</html>'
        ERRMSG(html,'Invalid argument')
def msg_help():
	html='<html>"<b>List of valid options (arguments) for "ufproc" <br>'
	html+='Valid arguments are: <br><br>'
	html+='<table><tr><td><font color="red">ufproc help</font> :</td><td>List of valid options (arguments) for ufproc</td></tr>'
	html+='<tr><td><font color="red">ufproc inter</font> :</td><td> Interleaved acquisition</td></tr>'
	html+='<tr><td><font color="red">ufproc jres</font> :</td><td> J-resolved acquisition</td></tr>'
	html+='<tr><td><font color="red">ufproc x</font> :</td><td>will apply the user defined time offset (x)</td></tr>'
	html+='<tr><td><font color="red">ufproc 0</font> :</td><td>will not apply any offset (x=0)</td></tr>'
	html+='<tr><td><font color="red">ufproc shon</font> :</td><td> will force program to apply shearing correction</td></tr>'
	html+='<tr><td><font color="red">ufproc abson</font> :</td><td>will apply the user defined baseline correction</td></tr>'
	html+='<tr><td><font color="red">ufproc dz</font> :</td><td>will remove both 1/8th extremities of the spectra for the time offset calculation</td></tr>'
	html+='<tr><td><font color="red">ufproc save</font> :</td><td> will save the 2D map of both positive and negative gradients (grad+ and grad-)</td></tr>'
	html+='<tr><td><font color="red">ufproc rot</font> :</td><td> will apply a rotation F1/F2 of the 2D map</td></tr>'
	html+='<tr><td><font color="red">ufproc +</font> :</td><td>will process only acquisition during positive gradients</td></tr></table><br></html>'
        MSG(html,'ufproc\'s options')
def msg_no2D(DIM):
        ERRMSG('Current data set is a '+str(DIM)+' dimension acquisition data','Incorrect dimension')
def msg_nodata():
        ERRMSG('No NMR data open or selected')
def msg_noUF(name,expno):
        ERRMSG(name+' / '+expno+' is not an Ultafast acquisition data')
def msg_badtopspinversion():
        ERRMSG('The script is not compatible with Topspin 1.x'+chr(10)+'Program exiting.')
def msg_deltree(rep):
        ERRMSG('Program can not delete the repertory "'+rep+'"\r\n\
        Try to delete it manually (maybe close Topspin first).')
def msg_copyfile(path,source):
        ERRMSG('Program can not save your original acquisition files\r\n\
        (Permission denied)\r\n\
        Execution is aborted.')
        
def msg_L3(_TD2):
        while 1:
                loop=INPUT_DIALOG('Loop selection gradients\
                ','"L3" is not the parameter that determines the loops of paired gradients.\
                ', ['Please specify the number of paired gradients'], [''], [''], ['1'],['Ok','Exit'])

       
                try:  value=int(loop[0])
                except:  MSG('Enter an integer for the loops of paired gradients.')
                else:
                        if value<=4:  MSG(str(value)+' is not a valid value for UltraFast NMR')
                        elif _TD2%value!=0:  MSG('TD 2 = '+str(_TD2)+' must be a multiple of the loops of paired gradients')
                        else:  break
        return value

def msg_type():
	html='<html>Select the type of acquisition: Interleaved or J-Resolved<br><br>'
	html+='<i>To avoid this message: run ufproc with the corresponding argument:</i><br>'
	html+='<table><tr><td><font color="red">ufproc jres</font> :</td><td>[J-Resolved], not necessary if the string "jres" is inside the pulsprog name</td></tr>'
	html+='<tr><td><font color="red">ufproc inter</font> :</td><td>[Interleaved], not necessary if the string "iuf" is inside the pulsprog name</td></tr>'
	html+='<tr><td><font color="red">ufproc</font> :</td><td>without argument will process as a one shot acquisition (TD 1 = 1)</td></tr></table></html>'
        return SELECT('',html,['Interleaved','J-Resolved','Exit'])
def msg_badSI(_SI,SImin,SImax,SIdefault,i):
        value=SELECT('','SI (F'+str(i)+') = '+str(_SI)+', is not suitable for zero filling.\r\n\
        SUGGESTION:\r\n\
        SI (F'+str(i)+') = '+str(SIdefault)+'\r\n\
        OK: Continue with SI'+str(i)+' = '+str(SIdefault)+'\r\n\
        Exit: exit the program.\r\n\
        ',['Ok','Exit'])
        return value

def msg_badphmod1():
        MSG('Program set the Phase correction mode to "magnitude calculation" (mc)')

def msg_badTDeff():
        MSG('Programm do not allow to reduce the number of points for fourrier transform \r\n \
	Program set TDeff=0 (all points) for both dimensions')
	
def msg_badmemod2():
        MSG('Linear prediction in F2 dimension is not suitable for ultrafast data\r\n \
	Program disabled ME_mod in F2')
	
def msg_save():
        MSG('UltraFast processing done \r\n \
        grad+- / 1 / 1 : 2D spectrum from positive gradients.\r\n\
        grad+- / 1 / 2 : 2D spectrum from negative gradients.')
        
def msg_done():
        MSG('UltraFast processing done')
        
def msg_exit():
        MSG('Program exiting','EXIT by the user')
        return 0  
def msg_error():
        ERRMSG('An error occured, Program exiting.')



#__________________________________________________MAIN PROGRAM_______________________________________________

        
sys.stdout.write('___________________________UFPROC START______________________________ \r\n')
#********************** Manage ufproc's arguments ******************************************************
arg = {'map':'NULL',
       'save':'NULL',
       'offs':'NULL',
       'shear':'shoff',
       'type':'oneshot',
       'abs':'NULL',
       'dzon' : 'NULL',
       'rot' : 'NULL'}
if len(sys.argv)>1:
	if 'help' in sys.argv[1:]:
		msg_help()
		EXIT()
        user_offset='NULL'
        for user_arg in sys.argv[1:]:
                if is_int(user_arg):
                        user_offset=user_arg
                if user_arg not in ['NULL','+','save',user_offset,'shon','shoff','jres','inter','abson','dz','rot','help']:
                        msg_badarg(user_arg)
                        EXIT()
                        break
                
        if user_offset in sys.argv[1:]: arg['offs']=user_offset
        if 'save' in sys.argv[1:]:      arg['save']='save'
        if '+' in sys.argv[1:]:
                arg['map']='+'
                arg['save']='NULL'
                arg['offs']='NULL'
        if 'shon' in sys.argv[1:]:      arg['shear']='shon'
        if 'shoff' in sys.argv[1:]:     arg['shear']='shoff'
        if 'jres' in sys.argv[1:]:      arg['type']='jres'
        if 'inter' in sys.argv[1:]:     arg['type']='inter'
	if 'abson' in sys.argv[1:]:    arg['abs']='abson'
	if 'dz' in sys.argv[1:]:    arg['dzon']=8
	if 'rot' in sys.argv[1:]:    arg['rot']='rot'
		
#******************************************To disable warning about file size (for topspin 4.x *************
XCMD("setdef ackn hide", WAIT_TILL_DONE)
#******************************************end*************************************************************

#******************************************end*************************************************************

#********************** Check if acquisition parameters are 2D NMR****************************************
DIM=GETACQUDIM()
if DIM==-1:
        msg_nodata()
        EXIT()
if DIM!=2:
        msg_no2D(DIM)
        EXIT()
dim=range(1,DIM+1)
#******************************************end*************************************************************
        
#********************** Topspin version & paths ***********************************************************************
curdata=CURDATA()
directory=curdata[3]
expno=curdata[1]
procno=curdata[2]
name=curdata[0]
if len(curdata)==5:   #Topspin version 1.x, 2.x
        try:dtypa=int(GETPARSTAT('DTYPA'))
        except:
               msg_badtopspinversion() #Topspin version 1.x
               EXIT()
        xwinversion=2        
        user=curdata[4]        
        PathName=directory+'/data/'+user+'/nmr/'+name
	PathGradName=directory+'/data/'+user+'/nmr/grad+-'

elif len(curdata)==4:   #Topspin version 3.x (no user)
        xwinversion=3
        user='nouser'        
        PathName=directory+'/'+name 
	PathGradName=directory+'/grad+-'	

PathExpno=PathName+'/'+expno
PathProcno=PathExpno+'/pdata/'+procno
PathGradExpno=['']
PathGradProcno=['']
for n in [1,2,3]:
	PathGradExpno.append(PathGradName+'/'+str(n))
	PathGradProcno.append(PathGradExpno[n]+'/pdata/1')


if name=='grad+-':
        msg_noUF(name,expno)
        EXIT()
#******************************************end*************************************************************

#************** Global variables **************************************************************************
size=4  #size of proc data
proc_files=['2rr','outd','proc','proc2','proc2s','procs']


#**************Clean files in case of error during the last execution****************************************
if os.path.isdir(PathGradName): deltree(PathGradName)
#******************************************end*************************************************************


#********************** Check the gradient acquisition loop (L3)*************************
_TD=(DIM+1)*[1]
for i in dim:
        _TD[i]=int(GETPARSTAT('TD',i))
        _TD[0]*=_TD[i]

L3=int(GETPARSTAT('L 3'))
if L3<=4 or _TD[2]%L3!=0:
        L3=msg_L3(_TD[2])
        if L3==-1:
                msg_exit()
                EXIT()
#******************************************end*************************************************************

#***Determine acquisition type ********************************************************************
if _TD[1]>1:
	if arg['type']=='oneshot': #no type argu
		pulprog=(GETPARSTAT('PULPROG'))
		if pulprog.lower().find('jres')!=-1:
			arg['type']='jres' #case jres written in TD1 shots
		elif pulprog.lower().find('iuf')!=-1:
			arg['type']='inter' #case interleaved 
		else:
			sel=msg_type()
			if sel==0:  arg['type']='inter'
			if sel==1:  arg['type']='jres'
			if sel==-1: 
				msg_exit()
				EXIT()

sys.stdout.write('Type of acquisition: '+arg['type']+'\r\n')
#******************************************end*************************************************************
 

#****************************************Check DIGTYP**************************************************
digtyp=int(GETPARSTAT('DIGTYP'))
sys.stdout.write('DIGTYP = '+str(digtyp)+'\r\n')
if digtyp==17:	
	with open(PathExpno+'/acqus','rt') as file:
		x = file.read()		
	with open(PathExpno+'/acqus','wt') as file:
		x = x.replace('DIGTYP= 17', 'DIGTYP= 12')
		file.write(x)
				
	with open(PathExpno+'/acqu','rt') as file:
		x = file.read()
	with open(PathExpno+'/acqu','wt') as file:
		x = x.replace('DIGTYP= 17', 'DIGTYP= 12')
		file.write(x)

	sys.stdout.write('DIGTYP changed from 17 (DRX) to 12 (DRU)\r\n')
#******************************************end*************************************************************


#******************************** check process parameters & determine acq parameters**********************
XCMD('wrpa grad+- 3')
disp('grad+-','3','1',directory,user)

_SI=(DIM+1)*[1]
for i in dim:
        _SI[i]=int(GETPAR('SI',i))
        _SI[0]*=_SI[i]

if arg['rot']=='rot' and _SI[1]!=_SI[2]:
    if _SI[1]>_SI[2]: _SI[2]=_SI[1]
    else:_SI[1]=_SI[2]

if arg['dzon']==8:
	DZ=8 #define Dead Zone: all spectra without 1/DZ points at both extremities
else:
	DZ=_SI[2]

_SWH2=float(GETPARSTAT('SWH',2))
DW=float(GETPARSTAT('DW'))
O=(DIM+1)*[0]
for i in dim: O[i]= float(GETPARSTAT('O'+str(i)))
WDW2=int(GETPAR('WDW',2))
PH_mod1=int(GETPAR('PH_mod',1))
ME_mod2=int(GETPAR('ME_mod',2))
TDeff1=int(GETPAR('TDeff',1))
TDeff2=int(GETPAR('TDeff',2))

TD=(DIM+1)*[1]

if arg['type']=='jres':
        TD[1]=_TD[1]/2
        TD[2]=_TD[2]
else:
	TD[1]=L3*_TD[1]
	TD[2]=_TD[2]/(2*L3)
for i in dim: TD[0]*=TD[i]

YSW_p=float(GETPARSTAT('SW_p',1))
XSW_p=float(GETPARSTAT('SW_p',2))
YOFFSET=float(GETPARSTAT('OFFSET',1))
XOFFSET=float(GETPARSTAT('OFFSET',2))

SI=(DIM+1)*[1]

SImin = (0,TD[1],TD[2]/2)
SImax = (0,32*TD[1],32*TD[2])
SIdefault = (0,2*TD[1],TD[2])

for i in dim:
        if _SI[i]>SImax[i] or _SI[i]<SImin[i]:
                if msg_badSI(_SI[i],SImin[i],SImax[i],SIdefault[i],i)==0:
                        PUTPAR(str(i)+' SI',str(SIdefault[i]))
                        _SI[i]=SIdefault[i]
                else:
                        msg_exit()
                        EXIT()
        
if PH_mod1!=2:
        msg_badphmod1()
        XCMD('1 PH_mod mc')
if ME_mod2!=0:
        msg_badmemod2()
        XCMD('2 ME_mod no')
if (TDeff1+TDeff2)!=0:
        msg_badTDeff()
	PUTPAR('1 TDeff','0')
	PUTPAR('2 TDeff','0')
#******************************************end*************************************************************

            
#****************** Write acqu status parameters as conventional 2D NMR ***********************************

SHOW_STATUS('UFPROC in progress...')

if arg['type']=='jres':
	SWH1 = 1000000*L3/(_TD[2]*_TD[1]*DW)
	SWH2=1/float(GETPARSTAT('AQ',2)) #approximation for jres
else:
	SWH1 = 1000000*_TD[1]*L3/(_TD[2]*DW) #inter
	SWH2 = SWH1/_TD[1] #oneshot+inter


list_ftmod=['no','isr','iqc','iqr','fsc','fsr','fqc','fqr','isc']
list_wdw=['no','EM','GM','SINE','QSINE','TRAP','USER','SINC','QSINC','TRAF','TRAFS']

#try:

PUTPAR('1 TD',str(TD[1]))
PUTPAR('1s TD',str(TD[1]))
PUTPAR('2 TD',str(TD[2]))
PUTPAR('2s TD',str(TD[2]))

PUTPAR('1 SWH',str(SWH1))
PUTPAR('1s SWH',str(SWH1))
PUTPAR('2 SWH',str(SWH2))
PUTPAR('2s SWH',str(SWH2))
#******************************************end**********************************************************************


#*********disable windows,phase, baseline correction, and linear prediction for next XF2()**************************
SI[1]=TD[1]
SI[2]=TD[2]/2


PUTPAR('2 WDW','no')
PUTPAR('2 PH_mod','no')
PUTPAR('2 BC_mod','no')		#no fid baseline correction on a spectra!!!!
PUTPAR('1 BC_mod','no')		#no fid baseline correction on a spectra!!!!

#******************************************end**********************************************************************


#**************************************** ser position zero ********************************************************
(size_ga,fmt_r_ga,fmt_w_ga) = fmt( int(GETPARSTAT('DTYPA')), int(GETPARSTAT('BYTORDA')) ) #format acqu grad+/-

digmod=int(GETPARSTAT('DIGMOD'))
sys.stdout.write('DIGMOD = '+str(digmod)+'\r\n')
sys.stdout.write('arg[type] = '+arg['type']+'\r\n')
if digmod==1 and arg['type']!='jres': #if digitization mode is digital & only one electronic delay
	DSPFVS=int(GETPARSTAT('DSPFVS'))
	sys.stdout.write('DSPFVS = '+str(DSPFVS)+'\r\n')
	if DSPFVS>=20:
		GRPDLY=int(float(GETPARSTAT('GRPDLY')))+1
		sys.stdout.write('GRPDLY = '+str(GRPDLY)+'\r\n')
		start=GRPDLY*2
		PUTPAR('2s DIGMOD','analog')	#put status digmod to analog
		PUTPAR('2 PKNL','false')	#not topspin to correct ser delay of digital filter
		ser=open(PathGradExpno[3]+'/ser','ab')
		write_bin(ser,fmt_w_ga,start*[0])
		ser.close()
	else:start=0
else:start=0
	
sys.stdout.write('ser position zero = '+str(start)+'\r\n')




#******************************************end*************************************************************


#++++++++++++++++++ write datas for each gradient as a conventional 2D map++++++++++++++++++++++++++++++
op=['positive','negative']
XCMD('wrpa grad+- 1')
if arg['map']=='+': n=1
else:
	n=2
	XCMD('wrpa grad+- 2')
	 
for grad in xrange(n): # positive:grad=0 expno=1 ;  negative:grad=1 expno=2
	disp('grad+-',str(grad+1),'1',directory,user)
	graddata=CURDATA()
	SHOW_STATUS('UFPROC in progress...')
	
	ser=open(PathGradExpno[3]+'/ser','rb')
	ser_g = open(PathGradExpno[grad+1]+'/ser','wb')

	if arg['type']!='inter':  #case one shot or jres
		block=TD[2]*size_ga  
		pos0=start*size_ga
		for td1 in xrange(TD[1]): #TD[1]
			ser.seek((td1*2+grad)*block+pos0)
			ser_li = ser.read(block)
			ser_g.write(ser_li[:])			
	
	else:	#case interleaved
		block_acq = _TD[2]*size_ga
		block_grad = TD[2]*size_ga
		pos0=start*size_ga
		
		for loop in xrange(L3) :
		    for n_acq in xrange(_TD[1]) :
			ser.seek(n_acq*block_acq+(2*loop+grad)*block_grad+pos0)
			ser_li = ser.read(block_grad)
			ser_g.write(ser_li[:])
	
	ser.close()
	ser_g.close()
	sys.stdout.write('program wrote ser of '+op[grad]+' gradients \r\n')
	
	# disable user's zero filling parameters:
	XCMD('.fid') #to avoid a 'can't display data' warning message 
		
	PUTPAR('1 SI',str(SI[1]))
	PUTPAR('2 SI',str(SI[2]))
	
	XF2()
	
	if grad==1: REV2()
	
	# Filtering in F2
	# active F2 user's process parameters:
	if xwinversion==2:      PUTPAR('2 WDW',str(WDW2))
	elif xwinversion==3:      PUTPAR('2 WDW',list_wdw[WDW2])
	PUTPAR('2 FT_mod','no')
	XTRFP2()	#process the filter in F2 without any transform   


	# active F2 user's zero filling parameters:
	XCMD('.fid')

	PUTPAR('2s SI',str(_SI[2]))
	PUTPAR('2 SI',str(_SI[2]))

	# zero filling of spectrum in F2 dimension
	(size_gp,fmt_r_gp,fmt_w_gp) = fmt( 'proc', int(GETPARSTAT('BYTORDP')) ) #format proc grad+/-
	rr = open(PathGradProcno[grad+1]+'/2rr','ab')
	ii = open(PathGradProcno[grad+1]+'/2ii','ab')
	write_bin(rr,fmt_w_gp,(TD[1]*(_SI[2]-SI[2]))*[0])
	write_bin(ii,fmt_w_gp,(TD[1]*(_SI[2]-SI[2]))*[0])
	rr.close()
	ii.close()

	num_ftmod=int(GETPARSTAT('FT_mod',2))
	FT_mod2=list_ftmod[num_ftmod]

	XIF2()
	
	#EXIT()    #uncomment to see data without FT in the conv. dim. (a deviation of the echoes indicates a shearing, can be corrected by a slight adjustment of the positive acquisition gradient 

	XCMD('2s FT_mod '+FT_mod2)
	if xwinversion==2:      PUTPAR('2s AXUNIT','')
	if xwinversion==3:      XCMD('2s AXUNIT >')


	Y=int(GETPARSTAT('XDIM',1))
	X=int(GETPARSTAT('XDIM',2))

	if arg['shear']!='shoff':
		
		dataset = top.BNMRDataSet.open(graddata)
		if grad==0:
			p_rr = dataset.getProcData().getDataDouble()
			p_ii = dataset.getProcData(dataconst.PROCDATA_IMAG).getDataDouble()
		else:
			n_rr = dataset.getProcData().getDataDouble()
			n_ii = dataset.getProcData(dataconst.PROCDATA_IMAG).getDataDouble()
		
		
	# active F1 user's zero filling parameters:
	XCMD('.fid')	
	PUTPAR('1s SI',str(_SI[1]))
	PUTPAR('1 SI',str(_SI[1]))

	XF1()

	PUTPAR('1s OFFSET',str(YOFFSET))

	if(arg['map']!='+'):
		XCMD('F2PROJP 1 '+str(_SI[1])+' 900')
		if grad==0:MAXp=int(GETPARSTAT('YMAX_p'))
		elif grad==1:MAXn=int(GETPARSTAT('YMAX_p'))
		(size_jp,fmt_r_jp,fmt_w_jp) = fmt( 'proc', int(GETPARSTAT('BYTORDP')) ) #format proc proj +/-
		r=open(PathGradExpno[grad+1]+'/pdata/900/1r','rb')
		if grad==0:projp_bin=r.read(_SI[2]*size)
		elif grad==1:projn_bin=r.read(_SI[2]*size)
		r.close()
		
	sys.stdout.write('2D map of '+op[grad]+' gradients complete \r\n')
	CLOSEWIN(CURDATA())
	if arg['save']!='save':CLOSEWIN(CURDATA())
	
#+++++++++++++++++++++++++++++++++++++++++++end+++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    


#***********calculate time offset between grad+ and grad- and apply on grad-*******************************

disp('grad+-','1','1',directory,user)
SHOW_STATUS('UFPROC in progress...(shearing calculation)')
#shearing calculation
if arg['shear']!='shoff':
	sys.stdout.write('grad+ : shearing calculation \r\n')
	action_p,list_shift_p=shearing(p_rr,p_ii,_SI[2],SI[1])
	if arg['map']!='+':
		if (arg['shear']=='shon' or action_p==1):
			sys.stdout.write('grad- : shearing calculation\r\n')
			action_n,list_shift_n=shearing(n_rr,n_ii,_SI[2],SI[1])
		else: action_n=0
	else: action_n=0  #case +
else: action_p=action_n=0 #case shoff

#offset calculation
if arg['map']!='+':
	if arg['offs']=='NULL':
		shift=offset(projp_bin,projn_bin,MAXp,MAXn,_SI[2],fmt_r_jp)
	else:   shift=int(user_offset)

else: shift=0 #case +

# offset and/or shearing apply
CLOSEWIN(CURDATA())
if action_p==1:                
	SHOW_STATUS('UFPROC in progress...(shift correction)')
	shift_corrections(PathGradProcno[1],X,Y,_SI[2],_SI[1],fmt_w_gp,0,list_shift_p)
	disp('grad+-','1','1',directory,user)
	PUTPAR('1s XDIM',str(_SI[1]))
	PUTPAR('2s XDIM',str(_SI[2]))
if action_n==1:
	
	SHOW_STATUS('UFPROC in progress...(shift correction)')
	shift_corrections(PathGradProcno[2],X,Y,_SI[2],_SI[1],fmt_w_gp,shift,list_shift_n)
	disp('grad+-','2','1',directory,user)
	PUTPAR('1s XDIM',str(_SI[1]))
	PUTPAR('2s XDIM',str(_SI[2]))
else:
	if (arg['offs']!=0 and arg['map']!='+' and shift!=0):
		
		SHOW_STATUS('UFPROC in progress...(shift correction)')
		shift_corrections(PathGradProcno[2],X,Y,_SI[2],_SI[1],fmt_w_gp,shift,[])
		disp('grad+-','2','1',directory,user)
		PUTPAR('1s XDIM',str(_SI[1]))
		PUTPAR('2s XDIM',str(_SI[2]))
	    
#------------------------------------------------end------------------------------------------------------


#*****************add2D positive map + negative map********************************************************
if arg['save']=='save':
	deltree(PathGradExpno[3])
	disp('grad+-','1','1',directory,user)
	XCMD('wrpa grad+- 3')
	disp('grad+-','3','1',directory,user)
else:
	disp('grad+-','1','1',directory,user)
	
SHOW_STATUS('UFPROC in progress...(Add maps +/-)')
if arg['map']!='+':
	edc2(directory,user,'grad+-','2','1',)
	PUTPAR('1 GAMMA',str(1.0))
	PUTPAR('2 GAMMA',str(1.0))
	PUTPAR('1 ALPHA',str(1.0))
	PUTPAR('2 ALPHA',str(1.0))
	ADD2D()
#*************************************End*****************************************************************

#************************** baseline correction********************************
if arg['abs']=='abson':
	PUTPAR('1 ABSF1',str(1000))
	PUTPAR('1 ABSF2',str(-1000))
	PUTPAR('2 ABSF1',str(1000))
	PUTPAR('2 ABSF2',str(-1000))

	ABS1()
	ABS2()
#********************** Apply calibration from edc*****************************
XCMD('2s sw_p '+str(XSW_p))
XCMD('1s sw_p '+str(YSW_p))
XCMD('2s OFFSET '+str(XOFFSET))
XCMD('1s OFFSET '+str(YOFFSET))

#********************** Copy results (2rr, proc files) from grad+-/3 to original dataset******************
if arg['save']=='save':
	for f in proc_files:
		shutil.copyfile(PathGradProcno[3]+'/'+f,PathProcno+'/'+f)
else:
	for f in proc_files:
		shutil.copyfile(PathGradProcno[1]+'/'+f,PathProcno+'/'+f)
disp(name,expno,procno,directory,user)


#*******************************Rotation of 2D map*******************************************************
#2D map datas are written per blocks (submatrix) from boottom left corner to the right side, then the upper horizontal line,etc.. 
#the last block is the top right corner.
if(arg['rot']=='rot'):
	Y=int(GETPARSTAT('XDIM',1))
	X=int(GETPARSTAT('XDIM',2))
	S=X*Y    #size of submatrix

	Nx=_SI[2]/X  #number of horizontal submatrix (F2)
	Ny=_SI[1]/Y  #number of vertical submatrix (F1)


	size_2rr=4 #4 bytes per data
	file2rr = open(PathProcno+'/2rr','rb') #original proc file in read binary mode
	file2rr_rot = open(PathProcno+'/2rr_rot','wb') #temporary file in write binary mode
	#CLOSEWIN(CURDATA()) #to prevent bug when reading file and display the map in the same time
	SHOW_STATUS('UFPROC in progress...')

	for Bx in range(Nx-1,-1,-1): #start with the right column
		for n in range(X):
			p0=Bx*S+X-1-n
			for By in range(Ny):			
				pointer=Nx*By*S+p0
				for i in range(Y):
					file2rr.seek((pointer+i*X)*size_2rr) #pointer position
					file2rr_rot.write(file2rr.read(size_2rr)) #write one binary data
				
	file2rr.close()
	file2rr_rot.close()
	os.remove(PathProcno+'/2rr')
	os.rename(PathProcno+'/2rr_rot', PathProcno+'/2rr')

	disp(name,expno,procno,directory,user)
	#the new file is written without submatrix
	PUTPAR('1s XDIM',str(_SI[1]))
	PUTPAR('2s XDIM',str(_SI[2]))				
	XCMD('REV1')	#reverse in F1 dimension
	disp(name,expno,procno,directory,user)
	sys.stdout.write('2D map is rotated' +'\r\n')



#XCMD('ufcalauto')
XCMD('.f1r')#zoom all
XCMD('.f2r')
t2 = time.clock()
#*************************************End*****************************************************************

sys.stdout.write('process duration:'+str(t2-t1)+'\r\n')

#*******delete temporary grad+- directory *************************************************
if arg['save']!='save':
	deltree(PathGradName)
else: msg_save()
#*************************************End*****************************************************************
XCMD("setdef ackn ok", WAIT_TILL_DONE)
msg_done()
#__________________________________________________END________________________________________________________________
